/***************************************************************
 *                   Materials Object Library                  *
 * Class isotropic_tensor : declaration for isotropic material *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.0.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2002,2003,2004,2005,2006,2007,2008,2009,2010,2012 COLLARD Christophe
 * copyright © 2006 PARGUEZ Cristelle
 * copyright © 2002,2003,2004,2005,2006,2007,2008,2009,2010,2012 Centre National de la Recherche Scientifique
 * copyright © 2002,2003,2004,2005,2006,2007,2008,2009,2010 Arts et Métiers ParisTech
 * copyright © 2002,2003,2004,2005,2006,2007 Université de Valenciennes et du Hainaut-Cambrésis
 * copyright © 2002,2003,2004,2005,2006,2007,2008,2009,2010 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2002,2003,2004,2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::isotropic_tensor
    \brief \f$ 4^{th} \f$ order isotropic tensor computation

    \htmlonly 
    <FONT color="#838383">

    isotropic tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    Let \f$ \lambda \f$ and \f$ \mu \f$ be two constants. We define a \f$ 4^{th} \f$ order isotropic tensor \f$ \boldsymbol{C} \f$ through \f$ \lambda \f$ and \f$ \mu \f$ by \n
    \f$ C_{ijkl} = \lambda \delta_{ij} \delta_{kl} + \mu (\delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk}) \f$

    \author copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; \endhtmlonly 2006 Cristelle PARGUEZ \n
            copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2007 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
            copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	    copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.0.0
    \date 2002-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type isotropic tensor
#endif

#ifndef __isotropic_tensors_hpp
#define __isotropic_tensors_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __fstream
#include <fstream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __math_h
#include <math.h>
#endif

#ifndef __string_h
#include <string.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __tensors2_hpp
#include "MOL++/tensors2.hpp"
#endif

#ifndef __symtensors2_hpp
#include "MOL++/symtensors2.hpp"
#endif

#ifndef __tensors4_hpp
#include "MOL++/tensors4.hpp"
#endif

#ifndef __symtensors4_hpp
#include "MOL++/symtensors4.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//=======================================
template <class T> class isotropic_tensor
//=======================================
{
  protected :
    T Lambda, Mu;
    string tensor_name;

  public :
    isotropic_tensor (string = "");  // default constructor
    isotropic_tensor (T, T, string = "");
    isotropic_tensor (const vector<T>&, string = "");
    isotropic_tensor (const tensor4<T>&, string = "");     // cast convertion
    isotropic_tensor (const symtensor4<T>&, string = "");  // cast convertion
    ~isotropic_tensor () {}  // destructor

    operator tensor4<T> () const; // cast operator : isotropic_tensor -> tensor4
    operator symtensor4<T> () const; // cast operator : isotropic_tensor -> symtensor4

    void name(string);  // computes tensor's name
    string name();      // returns tensor's name
    virtual T lambda (T);
    virtual T mu     (T);
    vector<T> elastic_constants ();

    // returns constants
    T lambda () const {return Lambda;}
    T mu     () const {return Mu;}

    friend bool operator ! (const isotropic_tensor<T>& c) {return !(c.Lambda && c.Mu);}

    virtual T operator () (int,int,int,int) const;    // returns the element Pijkl
    tensor3<T> operator [] (int) const; // returns a 3rd order tensor Cijkl -> Tjkl
    template <class Tf> friend isotropic_tensor<Tf> operator + (const isotropic_tensor<Tf>&, const isotropic_tensor<Tf>&);// overloads operator + for 4th order isotropic tensors
    template <class Tf> friend isotropic_tensor<Tf> operator - (const isotropic_tensor<Tf>&, const isotropic_tensor<Tf>&);// overloads operator - for 4th order isotropic tensors
    template <class Tf> friend Tf operator | (const isotropic_tensor<Tf>&, const isotropic_tensor<Tf>&);  // T_ijkl  R_ijkl
    template <class Tf> friend isotropic_tensor<Tf> operator || (const isotropic_tensor<Tf>&, const isotropic_tensor<Tf>&);  // P_ijkl = T_ijpq R_pqkl 
    isotropic_tensor<T>& operator += (const isotropic_tensor<T>&);  // overloads += operator for isotropic tensors
    isotropic_tensor<T>& operator -= (const isotropic_tensor<T>&);  // overloads -= operator for isotropic tensors

    // operations with scalars
    template <class Tf> friend isotropic_tensor<Tf> operator * (const isotropic_tensor<Tf>&, const Tf&);
    template <class Tf> friend isotropic_tensor<Tf> operator * (const Tf&, const isotropic_tensor<Tf>&);
    isotropic_tensor<T>& operator *= (const T&);

    // operations withs vectors
    template <class Tf> friend tensor3<Tf> operator * (const isotropic_tensor<Tf>&, const vector<Tf>&);  // P_ijk = T_ijkl * V_l
    template <class Tf> friend tensor3<Tf> operator * (const vector<Tf>&, const isotropic_tensor<Tf>&);  // P_jkl = T_ijkl * V_i
    template <class Tf> friend tensor3<Tf> operator | (const isotropic_tensor<Tf>&, const vector<Tf>&);  // P_ijl = T_ijkl * V_k
    template <class Tf> friend tensor3<Tf> operator | (const vector<Tf>&, const isotropic_tensor<Tf>&);  // P_ikl = T_ijkl * V_j
    template <class Tf> friend tensor3<Tf> tensor_sum (const isotropic_tensor<Tf>&, const vector<Tf>&, int);  // P = T : v

    // operation with 2nd order tensor
    template <class Tf> friend symtensor2<Tf> operator || (const isotropic_tensor<Tf>&, const tensor2<Tf>&);  // P_ij = T_ijkl * M_kl
    template <class Tf> friend symtensor2<Tf> operator || (const tensor2<Tf>&, const isotropic_tensor<Tf>&);  // P_kl = M_ij * T_ijkl

    // operation with 3rd order tensor
    template <class Tf> friend tensor3<Tf> operator || (const isotropic_tensor<Tf>&, const tensor3<Tf>&);  // P_n,ij = T_ijkl R_n,kl
    template <class Tf> friend tensor3<Tf> operator || (const tensor3<Tf>&, const isotropic_tensor<Tf>&);  // P_n,ij = R_n,kl T_klij

    // operations with 4rd order tensor
    template <class Tf> friend tensor4<Tf> operator + (const isotropic_tensor<Tf>&, const tensor4<Tf>&);
    template <class Tf> friend tensor4<Tf> operator + (const tensor4<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend tensor4<Tf> operator - (const isotropic_tensor<Tf>&, const tensor4<Tf>&);
    template <class Tf> friend tensor4<Tf> operator - (const tensor4<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend Tf operator | (const isotropic_tensor<Tf>&, const tensor4<Tf>&);
    template <class Tf> friend Tf operator | (const tensor4<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend tensor4<Tf> operator || (const isotropic_tensor<Tf>&, const tensor4<Tf>&);
    template <class Tf> friend tensor4<Tf> operator || (const tensor4<Tf>&, const isotropic_tensor<Tf>&);

    // operations with symmetric 4rd order tensor
    template <class Tf> friend symtensor4<Tf> operator + (const isotropic_tensor<Tf>&, const symtensor4<Tf>&);
    template <class Tf> friend symtensor4<Tf> operator + (const symtensor4<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend symtensor4<Tf> operator - (const isotropic_tensor<Tf>&, const symtensor4<Tf>&);
    template <class Tf> friend symtensor4<Tf> operator - (const symtensor4<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend Tf operator | (const isotropic_tensor<Tf>&, const symtensor4<Tf>&);
    template <class Tf> friend Tf operator | (const symtensor4<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend symtensor4<Tf> operator || (const isotropic_tensor<Tf>&, const symtensor4<Tf>&);
    template <class Tf> friend symtensor4<Tf> operator || (const symtensor4<Tf>&, const isotropic_tensor<Tf>&);

    template <class Tf> friend bool operator == (const isotropic_tensor<Tf>&, const isotropic_tensor<Tf>&);  // compares 2 isotropic tensors
    template <class Tf> friend bool operator != (const isotropic_tensor<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend bool operator == (const isotropic_tensor<Tf>&, const tensor4<Tf>&);
    template <class Tf> friend bool operator == (const tensor4<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend bool operator != (const isotropic_tensor<Tf>&, const tensor4<Tf>&);
    template <class Tf> friend bool operator != (const tensor4<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend bool operator == (const isotropic_tensor<Tf>&, const symtensor4<Tf>&);
    template <class Tf> friend bool operator == (const symtensor4<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend bool operator != (const isotropic_tensor<Tf>&, const symtensor4<Tf>&);
    template <class Tf> friend bool operator != (const symtensor4<Tf>&, const isotropic_tensor<Tf>&);

    isotropic_tensor<T> inv () const;
    template <class Tf> friend isotropic_tensor<Tf> change_orthonormal_basis (const matrix<Tf>&, const isotropic_tensor<Tf>&);
    template <class Tf> friend tensor4<Tf> change_basis (const matrix<Tf>&, const isotropic_tensor<Tf>&);

    template <class Tf> friend ostream& operator << (ostream&, const isotropic_tensor<Tf>&);
    template <class Tf> friend istream& operator >> (istream&, isotropic_tensor<Tf>&);
    virtual void save (const char*); // writes to disk
    virtual void read (const char*); // reads from disk
    isotropic_tensor<T>& approximation ();
};


//=====Private methods for isotropic tensor=============================================


//=====Public methods for isotropic tensor==============================================

/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  \param name tensor name
*/

//--------------------------------------------------------------------
template <class T> isotropic_tensor<T>::isotropic_tensor (string name)
//--------------------------------------------------------------------
{
  Lambda = Mu = 0;
  tensor_name = name;
}


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  \f$ C_{ijkl} = \lambda \delta_{ij} \delta_{kl} + \mu (\delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk}) \f$.

  \param lambda \f$ \lambda \f$ value
  \param mu \f$ \mu \f$ value
  \param name tensor name
*/

//------------------------------------------------------------------------------------
template <class T> isotropic_tensor<T>::isotropic_tensor (T lambda, T mu, string name)
//------------------------------------------------------------------------------------
{
  Lambda = lambda;
  Mu = mu;
  tensor_name = name;
}


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  \f$ C_{ijkl} = \lambda \delta_{ij} \delta_{kl} + \mu (\delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk}) \f$.

  \param v \f$ (\lambda,\mu) \f$ values
  \param name tensor name
*/

//----------------------------------------------------------------------------------------
template <class T> isotropic_tensor<T>::isotropic_tensor (const vector<T>& v, string name)
//----------------------------------------------------------------------------------------
{
  assert (v.dim() == 2);

  Lambda = v[1];
  Mu = v[2];
  tensor_name = name;
}


/*!
  \brief Constructor for cast conversion.

  Converts a \f$ \displaystyle 4^{th} \f$ order tensor into an isotropic tensor.

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param name tensor name
*/

//-----------------------------------------------------------------------------------------
template <class T> isotropic_tensor<T>::isotropic_tensor (const tensor4<T>& t, string name)
//-----------------------------------------------------------------------------------------
{
  assert(t.dim1()==3 && t.dim2()==3 && t.dim3()==3 && t.dim4()==3);

  Lambda = t(1,1,2,2);
  Mu = t(1,2,1,2);
  tensor_name = name;
  assert ((tensor4<T>)(*this) == t);
}


/*!
  \brief Constructor for cast conversion.

  Converts a symmetric \f$ \displaystyle 4^{th} \f$ order tensor into an isotropic tensor.

  \param st symmetric \f$ \displaystyle 4^{th} \f$ order tensor
  \param name tensor name
*/

//---------------------------------------------------------------------------------------------
template <class T> isotropic_tensor<T>::isotropic_tensor (const symtensor4<T>& st, string name)
//---------------------------------------------------------------------------------------------
{
  assert(st.dim1()==3 && st.dim3()==3);

  Lambda = st(1,1,2,2);
  Mu = st(1,2,1,2);
  tensor_name = name;
  assert ((tensor4<T>)(*this) == st);
}


/*!
  \brief Cast operator for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  Converts an isotropic tensor into a \f$ \displaystyle 4^{th} \f$ order tensor.
*/

//------------------------------------------------------------------
template <class T> isotropic_tensor<T>::operator tensor4<T> () const
//------------------------------------------------------------------
{
  assert(Lambda && Mu);

  tensor4<T> tsr;
  tsr.assign(3,3,3,3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  tsr(i,j,k,l) = Lambda * (i==j) * (k==l) + Mu * ( (i==k) * (j==l) + (i==l) * (j==k) );

  return tsr;
}



/*!
  \brief Cast operator for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  Converts an isotropic tensor into a symmetric \f$ \displaystyle 4^{th} \f$ order tensor.
*/

//---------------------------------------------------------------------
template <class T> isotropic_tensor<T>::operator symtensor4<T> () const
//---------------------------------------------------------------------
{
  assert(Lambda && Mu);

  symtensor4<T> tsr (3);

  for (int i=1; i<=3; i++)
    { tsr (i,i,i,i) = 2 * Mu;
      for (int j=1; j<=3; j++)
	tsr (i,i,j,j) += Lambda;
    }

  tsr (1,2,1,2) = tsr (1,3,1,3) = tsr (2,3,2,3) = Mu;

  return tsr;
}



/*!
  \brief Changes tensor name

  \param name tensor name
*/

//-------------------------------------------------------------
template <class T> void isotropic_tensor<T>::name (string name)
//-------------------------------------------------------------
{
  tensor_name = name;
}


/*!
  \brief Returns tensor name

  \return tensor name
*/

//----------------------------------------------------
template <class T> string isotropic_tensor<T>::name ()
//----------------------------------------------------
{
  return tensor_name;
}


/*!
  \brief Changes and returns \f$ \lambda \f$ value (see detailed description).
  \param lambda \f$ \lambda \f$ value
  \return \f$ \lambda \f$
*/

//---------------------------------------------------------
template <class T> T isotropic_tensor<T>::lambda (T lambda)
//---------------------------------------------------------
{
  return Lambda = lambda;
}


/*!
  \brief Changes and returns \f$ \mu \f$ value (see detailed description).
  \param mu \f$ \mu \f$ value
  \return \f$ \mu \f$
*/

//-------------------------------------------------
template <class T> T isotropic_tensor<T>::mu (T mu)
//-------------------------------------------------
{
  return Mu = mu;
}


/*!
  \brief Returns elastic constants.
  \return \f$ (\lambda, \mu) \f$
*/

//--------------------------------------------------------------------
template <class T> vector<T> isotropic_tensor<T>::elastic_constants ()
//--------------------------------------------------------------------
{
  vector<T> lame_constants (2,false);
  lame_constants[1] = Lambda;
  lame_constants[2] = Mu;

  return lame_constants;
}


/*!
  \brief Returns an element of a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ C \f$ be a \f$ \displaystyle 4^{th} \f$ order isotropic tensors defined through \f$ (\lambda,\mu) \f$. We compute \n
 \f$ C_{ijkl} = \lambda \delta_{ij} \delta_{kl} + \mu (\delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk}) \f$.

  \param i,j,k,l coordinates
  \return scalar \f$ C_{ijkl} \f$

*/

//-----------------------------------------------------------------------------------
template <class T> T isotropic_tensor<T>::operator () (int i,int j,int k,int l) const
//-----------------------------------------------------------------------------------
{
  assert ((i>0 && i<=3) && (j>0 && j<=3) && (k>0 && k<=3) && (l>0 && l<=3));
  assert (Lambda && Mu);

  return  Lambda * (i==j) * (k==l) + Mu * ( (i==k) * (j==l) + (i==l) * (j==k));
}


/*!
  \brief Returns the \f$ i^\text{th} \f$ \f$ \displaystyle 3^{rd} \f$ order tensor of a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ C \f$ be a \f$ \displaystyle 4^{th} \f$ order isotropic tensor defined through \f$ (\lambda,\mu) \f$ and \f$ R \f$ be a \f$ \displaystyle 3^{rd} \f$ order tensor. For a constant component \f$ i \f$, we compute \n
  \f$ R_{jkl} = C_{ijkl} \f$.

  \param i coordinate
  \return \f$ \displaystyle 3^{rd} \f$ order tensor
  \note Fonction defined for compatibility with \f$ 4^\text{th} \f$ order tensors.

*/

//--------------------------------------------------------------------------
template <class T> tensor3<T> isotropic_tensor<T>::operator [] (int i) const
//--------------------------------------------------------------------------
{
  assert (i>0 && i<=3);
  assert (Lambda && Mu);

  tensor3<T> tsr3;
  tsr3.assign(3,3,3);
  for (int j=1; j<=3; j++)
    for (int k=1; k<=3; k++)
      for (int l=1; l<=3; l++)
	tsr3(j,k,l) = Lambda * (i==j) * (k==l) + Mu * ( (i==k) * (j==l) + (i==l) * (j==k) );

  return tsr3;
}


/*!
  \brief Addition for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  Let \f$ C_1 \f$ and \f$ C_2 \f$ be two \f$ \displaystyle 4^{th} \f$ order isotropic tensors defined respectively through \f$ (\lambda_1,\mu_1) \f$ and \f$ (\lambda_2,\mu_2) \f$.\n
  We compute \f$ C = C_1 + C_2 \f$ where \f$ C \f$ is a \f$ \displaystyle 4^{th} \f$ order isotropic tensor defined through \f$ (\lambda,\mu) \f$ with :\n
  \f$ \lambda=\lambda_1 + \lambda_2 \f$ \n
  \f$ \mu=\mu_1 + \mu_2 \f$

  \param C1 \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param C2 \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 4^{th} \f$ order isotropic tensor \f$ C \f$
*/

//------------------------------------------------------------------------------------------------------------------
template <class Tf> isotropic_tensor<Tf> operator + (const isotropic_tensor<Tf>& C1, const isotropic_tensor<Tf>& C2)
//------------------------------------------------------------------------------------------------------------------
{
  return isotropic_tensor<Tf> (C1.Lambda+C2.Lambda,C1.Mu+C2.Mu);
}


/*!
  \brief Subtraction for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  Let \f$ C_1 \f$ and \f$ C_2 \f$ be two \f$ \displaystyle 4^{th} \f$ order isotropic tensors defined respectively through \f$ (\lambda_1,\mu_1) \f$ and \f$ (\lambda_2,\mu_2) \f$.\n
  We compute \f$ C = C_1 - C_2 \f$ where \f$ C \f$ is a \f$ \displaystyle 4^{th} \f$ order isotropic tensor defined through \f$ (\lambda,\mu) \f$ with :\n
  \f$ \lambda=\lambda_1 - \lambda_2 \f$ \n
  \f$ \mu=\mu_1 - \mu_2 \f$

  \param C1 \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param C2 \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 4^{th} \f$ order isotropic tensor \f$ C \f$
*/

//------------------------------------------------------------------------------------------------------------------
template <class Tf> isotropic_tensor<Tf> operator - (const isotropic_tensor<Tf>& C1, const isotropic_tensor<Tf>& C2)
//------------------------------------------------------------------------------------------------------------------
{
  return isotropic_tensor<Tf> (C1.Lambda-C2.Lambda,C1.Mu-C2.Mu);
}


/*!
  \brief Doubly contracted product for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  We compute the doubly contracted product for the \f$ \displaystyle 4^{th} \f$ order isotropic tensors \f$ C^1 \f$ and \f$ C^2 \f$ defined respectively through \f$ (\lambda_1,\mu_1) \f$ and \f$ (\lambda_2,\mu_2) \f$ :\n
  \f$ C^1 :: C^2 = C^1_{ijkl}  C^2_{ijkl} = (3 \lambda_1 + 2 \mu_1)(3 \lambda_2 + 2 \mu_2) + 20 \mu_1 \mu_2 \f$.

  \param C1 \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param C2 \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return scalar \f$ C^1 :: C^2 \f$
*/

//------------------------------------------------------------------------------------------------
template <class Tf> Tf operator | (const isotropic_tensor<Tf>& C1, const isotropic_tensor<Tf>& C2)
//------------------------------------------------------------------------------------------------
{
  return ((3*C1.Lambda+2*C1.Mu)*(3*C2.Lambda+2*C2.Mu) + 20*C1.Mu*C2.Mu);
}


/*!
  \brief Contracted product for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  We compute the contracted product for the \f$ \displaystyle 4^{th} \f$ order isotropic tensors \f$ C^1 \f$ and \f$ C^2 \f$ :\n
  \f$ \left( C^1 : C^2 \right)_{ijkl} = C^1_{ijpq} C^2_{pqkl} \f$

  \param C1 \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param C2 \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 4^{th} \f$ order isotropic tensor \f$ C^1 : C^2 \f$
*/

//-------------------------------------------------------------------------------------------------------------------
template <class Tf> isotropic_tensor<Tf> operator || (const isotropic_tensor<Tf>& C1, const isotropic_tensor<Tf>& C2)
//-------------------------------------------------------------------------------------------------------------------
{
  return isotropic_tensor<Tf> ((C2.Lambda*(C1.Lambda+2*C1.Mu)+C1.Lambda*(C2.Lambda+2*C2.Mu)+C1.Lambda*C2.Lambda) , (2*C1.Mu*C2.Mu));
}


/*!
  \brief Operator += for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  Let \f$ N \f$ and \f$ C \f$ be two \f$ \displaystyle 4^{th} \f$ order isotropic tensors defined respectively through \f$ (\lambda,\mu) \f$ and \f$ (\lambda_c,\mu_c) \f$. We compute \f$ N += C \f$ in the following way : \n
  \f$ \lambda += \lambda_c \f$ \n
  \f$ \mu+ = \mu_c \f$

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 4^{th} \f$ order isotropic tensor \f$ N \f$
*/

//-----------------------------------------------------------------------------------------------------
template <class T> isotropic_tensor<T>& isotropic_tensor<T>::operator += (const isotropic_tensor<T>& C)
//-----------------------------------------------------------------------------------------------------
{
  assert (Lambda && Mu);

  Lambda += C.Lambda;
  Mu += C.Mu;

  return *this;
}


/*!
  \brief Operator -= for \f$ \displaystyle 4^{th} \f$ order isotropic tensors.

  Let \f$ N \f$ and \f$ C \f$ be two \f$ \displaystyle 4^{th} \f$ order isotropic tensors defined respectively through \f$ (\lambda,\mu) \f$ and \f$ (\lambda_c,\mu_c) \f$. We compute \f$ N -= C \f$ in the following way : \n
  \f$ \lambda -= \lambda_c \f$ \n
  \f$ \mu -= \mu_c \f$

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 4^{th} \f$ order isotropic tensor \f$ N \f$
*/

//-----------------------------------------------------------------------------------------------------
template <class T> isotropic_tensor<T>& isotropic_tensor<T>::operator -= (const isotropic_tensor<T>& C)
//-----------------------------------------------------------------------------------------------------
{
  assert (Lambda && Mu);

  Lambda -= C.Lambda;
  Mu -= C.Mu;

  return *this;
}


/*!
  \brief Product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a scalar.

  Let \f$ N \f$ and \f$ C \f$ be two \f$ \displaystyle 4^{th} \f$ order isotropic tensor defined respectively through \f$ (\lambda,\mu) \f$ and \f$ (\lambda_c,\mu_c) \f$.\n
  We compute the product \f$ N = C * k \f$ in the following way :\n
  \f$ \lambda = \lambda_c * k \f$ \n
  \f$ \mu = \mu_c * k \f$

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param elt scalar \f$ k \f$

  \return \f$ \displaystyle 4^{th} \f$ order isotropic tensor \f$ N \f$
*/

//------------------------------------------------------------------------------------------------
template <class Tf> isotropic_tensor<Tf> operator * (const isotropic_tensor<Tf>& C, const Tf& elt)
//------------------------------------------------------------------------------------------------
{
  return isotropic_tensor<Tf> (C.Lambda*elt,C.Mu*elt);
}


/*!
  \brief Product between a scalar and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ N \f$ and \f$ C \f$ be two \f$ \displaystyle 4^{th} \f$ order isotropic tensor defined respectively through \f$ (\lambda,\mu) \f$ and \f$ (\lambda_c,\mu_c) \f$.\n
  We compute the product \f$ N = k * C \f$ in the following way :\n
  \f$ \lambda = \lambda_c * k \f$ \n
  \f$ \mu = \mu_c * k \f$

  \param elt scalar \f$ k \f$
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 4^{th} \f$ order isotropic tensor \f$ N \f$
*/

//------------------------------------------------------------------------------------------------
template <class Tf> isotropic_tensor<Tf> operator * (const Tf& elt, const isotropic_tensor<Tf>& C)
//------------------------------------------------------------------------------------------------
{
  return isotropic_tensor<Tf> (C.Lambda*elt,C.Mu*elt);
}


/*!
  \brief Computes the product with a scalar.

  Let \f$ k \f$ be a scalar and \f$ C \f$ be a \f$ \displaystyle 4^{th} \f$ order isotropic tensor defined through \f$ (\lambda, \mu) \f$. \n
  We define \f$ C *= k \f$ by computing:\n
  \f$ \lambda *= k \f$ \n
  \f$ \mu *= k \f$

  \param elt scalar k

  \return \f$ \displaystyle 4^{th} \f$ order isotropic tensor \f$ C \f$
*/

//-------------------------------------------------------------------------------------
template <class T> isotropic_tensor<T>& isotropic_tensor<T>::operator *= (const T& elt)
//-------------------------------------------------------------------------------------
{
  assert (Lambda && Mu);

  Lambda *= elt;
  Mu *= elt;

  return *this;
}


/*!
  \brief Product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a vector.

  Let \f$ R \f$ be a \f$ \displaystyle 3^{rd} \f$ order tensor. We compute \f$ R = C * v \f$ :\n
  \f$ R_{ijk}=C_{ijkl} * v_l \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param v vector

  \return \f$ \displaystyle 3^{rd} \f$ order tensor \f$ R \f$
*/

//---------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator * (const isotropic_tensor<Tf>& C, const vector<Tf>& v)
//---------------------------------------------------------------------------------------------
{
  assert (v.dim() == 3);

  tensor3<Tf> tsr3(3,3,3,false);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	tsr3(i,j,k) = v[k] * C.Lambda * (i==j) + v[j] * C.Mu * (i==k) + v[i] * C.Mu * (j==k);

  return tsr3;
}


/*!
  \brief Product between a vector and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ R \f$ be a \f$ \displaystyle 3^{rd} \f$ order tensor. We compute \f$ R = v * C \f$ :\n
  \f$ R_{jkl} = v_i * C_{ijkl} \f$.

  \param v vector
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 3^{rd} \f$ order tensor \f$ R \f$
*/

//---------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator * (const vector<Tf>& v, const isotropic_tensor<Tf>& C)
//---------------------------------------------------------------------------------------------
{
  assert (v.dim() == 3);

  tensor3<Tf> tsr3(3,3,3,false);
  for (int j=1; j<=3; j++)
    for (int k=1; k<=3; k++)
      for (int l=1; l<=3; l++)
	tsr3(j,k,l) = v[j] * C.Lambda * (k==l)+ v[k] * C.Mu * (j==l)+ v[l] * C.Mu * (j==k);

  return tsr3;
}



/*!
  \brief   Product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a vector.

  Let \f$ R \f$ be a \f$ \displaystyle 3^{rd} \f$ order tensor. We compute \f$ R = C | v \f$ : \n
  \f$ R_{ijl} = C_{ijkl} * v_k = C_{ijlk} v_k \f$, so we get \f$ C | v = C * v \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param v vector

  \return \f$ \displaystyle 3^{rd} \f$ order tensor \f$ R \f$
*/

//---------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator | (const isotropic_tensor<Tf>& C, const vector<Tf>& v)
//---------------------------------------------------------------------------------------------
{
  return C*v;
}


/*!
  \brief Product between a vector and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ R \f$ be a \f$ \displaystyle 3^{rd} \f$ order tensor. We compute \f$ R = v | C \f$ : \n
  \f$ R_{ikl} = v_j * C_{ijkl} = v_j * C_{jikl} = v * C \f$, so we get \f$ v | C = v * C \f$.

  \param v vector
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 3^{rd} \f$ order tensor \f$ R \f$
*/

//---------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator | (const vector<Tf>& v, const isotropic_tensor<Tf>& C)
//---------------------------------------------------------------------------------------------
{
  return v*C;
}


/*!
  \brief Product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a vector.

  If \f$ i = 1,2 \f$ we compute v*C. \n
  If \f$ i = 3,4 \f$ we compute C*v. \n

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param v vector
  \param i \f$ \displaystyle 4^{th} \f$ order isotropic tensor coordinate which will be multiplied by the vector

  \return \f$ \displaystyle 3^{rd} \f$ order tensor
*/

//----------------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> tensor_sum (const isotropic_tensor<Tf>& C, const vector<Tf>& v, int i)
//----------------------------------------------------------------------------------------------------
{
  assert (i>=1 && i<=4);

  tensor3<Tf> tsr3;
  if (i==1 || i==2) tsr3 =& (v*C);
  else tsr3 =& (C*v);

  return tsr3;
}


/*!
  \brief Contracted product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a \f$ \displaystyle 2^{nd} \f$ order tensor.

  Let \f$ Q \f$ be a \f$ \displaystyle 2^{nd} \f$ order tensor. We compute \f$ Q = C : t \f$ \n
  \f$ Q_{ij} = C_{ijkl} * t_{kl} \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t \f$ \displaystyle 2^{nd} \f$ order tensor

  \return \f$ \displaystyle 2^{nd} \f$ order tensor \f$ Q \f$
*/

//--------------------------------------------------------------------------------------------------
template <class Tf> symtensor2<Tf> operator || (const isotropic_tensor<Tf>& C, const tensor2<Tf>& t)
//--------------------------------------------------------------------------------------------------
{
  assert (t.Rows() == 3);
  assert (t.Columns() == 3);

  symtensor2<Tf> mat_result (3, 3, false);

  Tf val = C.Lambda * tr(t);
  for (int i=1; i<=3; i++)
    { for (int j=i; j<=3; j++)
	mat_result(i,j) = C.Mu * (t(i,j) + t(j,i));
      mat_result(i,i) += val;
    }

  return mat_result;
}


/*!
  \brief Contracted product between a \f$ \displaystyle 2^{nd} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ Q \f$ be a \f$ \displaystyle 2^{nd} \f$ order tensor. We compute \f$ Q = t : C \f$ \n
  \f$ Q_{kl} = t_{ij} * C_{ijkl} = C_{klij} * t_{ij} \f$, so we get \f$ t : C = C : t \f$.

  \param t \f$ \displaystyle 2^{nd} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 2^{nd} \f$ order tensor \f$ Q \f$
*/

//--------------------------------------------------------------------------------------------------
template <class Tf> symtensor2<Tf> operator || (const tensor2<Tf>& t, const isotropic_tensor<Tf>& C)
//--------------------------------------------------------------------------------------------------
{
  return (C || t);
}


/*!
  \brief Contracted product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a \f$ \displaystyle 3^{rd} \f$ order tensor.

  Let W be a \f$ \displaystyle 3^{rd} \f$ order tensor. We compute \f$ W = C : t \f$ \n
  \f$ W_{n,ij} = C_{ijkl} * t_{n,kl} \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t \f$ \displaystyle 3^{rd} \f$ order tensor

  \return \f$ \displaystyle 3^{rd} \f$ order tensor \f$ W \f$
*/

//-----------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator || (const isotropic_tensor<Tf>& C, const tensor3<Tf>& t)
//-----------------------------------------------------------------------------------------------
{
  assert ( (t.dim1() == 3) && (t.dim2() == 3) && (t.dim3() == 3) );

  tensor3<Tf> ts;
  ts.create(3);
  for (int n=1; n<=3; n++)
    ts[n] = (t[n] || C);

  return ts;

}


/*!
  \brief Contracted product between a \f$ \displaystyle 3^{rd} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let W be a \f$ \displaystyle 3^{rd} \f$ order tensor. We compute \f$ W = t : C \f$ \n
  \f$ W_{n,kl} = t_{n,ij} * C_{ijkl} = C_{klij} * t_{n,ij} \f$, so we get \f$ t:C = C:t \f$.

  \param t \f$ \displaystyle 3^{rd} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 3^{rd} \f$ order tensor \f$ W \f$
*/

//-----------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator || (const tensor3<Tf>& t, const isotropic_tensor<Tf>& C)
//-----------------------------------------------------------------------------------------------
{
  return (C || t);
}


/*!
  \brief Addition between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a \f$ \displaystyle 4^{th} \f$ order tensor.

  Let \f$ N \f$ be a \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = C+t \f$ \n
  \f$ N_{ijkl} = C_{ijkl} + t_{ijkl} \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t \f$ \displaystyle 4^{th} \f$ order tensor

  \return \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//----------------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator + (const isotropic_tensor<Tf>& C, const tensor4<Tf>& t)
//----------------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && (t.dim2() == 3) && (t.dim3() == 3) && (t.dim4() == 3));

  tensor4<Tf> tsr(t);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      { tsr(i,i,j,j)+=C.Lambda;
	tsr(i,j,i,j)+=C.Mu;
	tsr(i,j,j,i)+=C.Mu;
      }

  return tsr;
}


/*!
  \brief Addition between a \f$ \displaystyle 4^{th} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ N \f$ be a \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = t+C \f$ \n
  \f$ N_{ijkl} = C_{ijkl} + t_{ijkl} \f$.

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//----------------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator + (const tensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//----------------------------------------------------------------------------------------------
{
  return (C + t);
}


/*!
  \brief Subtraction between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a \f$ \displaystyle 4^{th} \f$ order tensor.

  Let \f$ N \f$ be a \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = C-t \f$ \n
  \f$ N_{ijkl} = C_{ijkl} - t_{ijkl} \f$

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t \f$ \displaystyle 4^{th} \f$ order tensor

  \return \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//----------------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator - (const isotropic_tensor<Tf>& C, const tensor4<Tf>& t)
//----------------------------------------------------------------------------------------------

{
  assert (t.dim1() == 3 && (t.dim2() == 3) && (t.dim3() == 3) && (t.dim4() == 3));

  tensor4<Tf> tsr((Tf)-1*t);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      { tsr(i,i,j,j)+=C.Lambda;
	tsr(i,j,i,j)+=C.Mu;
	tsr(i,j,j,i)+=C.Mu;
      }

  return tsr;
}


/*!
  \brief Subtraction between a \f$ \displaystyle 4^{th} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ N \f$ be a \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = t-C \f$ \n
  \f$ N_{ijkl} = t_{ijkl} - C_{ijkl} \f$.

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//----------------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator - (const tensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//----------------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && (t.dim2() == 3) && (t.dim3() == 3) && (t.dim4() == 3));

  tensor4<Tf> tsr(t);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      { tsr(i,i,j,j)-=C.Lambda;
	tsr(i,j,i,j)-=C.Mu;
	tsr(i,j,j,i)-=C.Mu;
      }

  return tsr;
}


/*!
  \brief Doubly contracted product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a \f$ \displaystyle 4^{th} \f$ order tensor.

  We compute \f$ C :: t = C_{ijkl} * t_{ijkl} \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t \f$ \displaystyle 4^{th} \f$ order tensor

  \return scalar \f$ C :: t \f$
*/

//-------------------------------------------------------------------------------------
template <class Tf> Tf operator | (const isotropic_tensor<Tf>& C, const tensor4<Tf>& t)
//-------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && (t.dim2() == 3) && (t.dim3() == 3) && (t.dim4() == 3));

  Tf sum=0;
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      { sum += C.Lambda * t(i,i,j,j);
	sum += C.Mu * t(i,j,i,j);
	sum += C.Mu * t(i,j,j,i);
      }

  return sum;
}


/*!
  \brief Doubly contracted product between a \f$ \displaystyle 4^{th} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  We compute \f$ t :: C = C_{ijkl} * t_{ijkl} \f$.

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return scalar \f$ t :: C \f$
*/

//-------------------------------------------------------------------------------------
template <class Tf> Tf operator | (const tensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//-------------------------------------------------------------------------------------
{
  return (C|t);
}


/*!
  \brief Contracted product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a \f$ \displaystyle 4^{th} \f$ order tensor.

  Let \f$ N \f$ be a \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = C : t \f$ \n
  \f$ N_{ijkl} = C_{ijpq} * t_{pqkl} \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t \f$ \displaystyle 4^{th} \f$ order tensor

  \return  \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//-----------------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator || (const isotropic_tensor<Tf>& C, const tensor4<Tf>& t)
//-----------------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && (t.dim2() == 3) && (t.dim3() == 3) && (t.dim4() == 3));

  tensor4<Tf> tsr(3,3,3,3,false);
  for (int i=1; i<=3; i++)
    for (int p=1; p<=3; p++)
      for (int q=1; q<=3; q++)
      { for (int j=1; j<=3; j++)
	  tsr(i,j,p,q) = C.Mu*(t(i,j,p,q)+t(j,i,p,q));
	for (int k=1; k<=3; k++)
	  tsr(i,i,p,q) += C.Lambda*t(k,k,p,q);
      }

  return tsr;
}


/*!
  \brief Contracted product between a \f$ \displaystyle 4^{th} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ N \f$ be a \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = t : C \f$ \n
  \f$ N_{ijkl} = t_{ijpq} * C_{pqkl} \f$.

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return  \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//-----------------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> operator || (const tensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//-----------------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && (t.dim2() == 3) && (t.dim3() == 3) && (t.dim4() == 3));

  tensor4<Tf> tsr(3,3,3,3,false);

  for (int p=1; p<=3; p++)
    for (int q=1; q<=3; q++)
      for (int k=1; k<=3; k++)
	{ for (int l=1; l<=3; l++)
	    tsr(p,q,k,l) = C.Mu*(t(p,q,k,l)+t(p,q,l,k));
	  for (int i=1; i<=3; i++)
	    tsr(p,q,k,k) += C.Lambda*t(p,q,i,i);
	}

  return tsr;
}


/*!
  \brief Addition between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a symmetric \f$ \displaystyle 4^{th} \f$ order tensor.

  Let \f$ N \f$ be a symmetric \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = C+t \f$ \n
  \f$ N_{ijkl} = C_{ijkl} + t_{ijkl} \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t \f$ symmetric \displaystyle 4^{th} \f$ order tensor

  \return symmetric \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//----------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator + (const isotropic_tensor<Tf>& C, const symtensor4<Tf>& t)
//----------------------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && t.dim3() == 3);

  symtensor4<Tf> tsr (t);

  for (int i=1; i<=3; i++)
    { tsr (i,i,i,i) += 2 * C.Mu;
      for (int j=1; j<=3; j++)
	tsr (i,i,j,j) += C.Lambda;
    }

  tsr (1,2,1,2) += C.Mu;
  tsr (1,3,1,3) += C.Mu;
  tsr (2,3,2,3) += C.Mu;

  return tsr;
}


/*!
  \brief Addition between a symmetric \f$ \displaystyle 4^{th} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ N \f$ be a symmetric \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = t+C \f$ \n
  \f$ N_{ijkl} = C_{ijkl} + t_{ijkl} \f$.

  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return symmetric \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//----------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator + (const symtensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//----------------------------------------------------------------------------------------------------
{
  return (C + t);
}


/*!
  \brief Subtraction between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a symmetric \f$ \displaystyle 4^{th} \f$ order tensor.

  Let \f$ N \f$ be a symmetric \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = C-t \f$ \n
  \f$ N_{ijkl} = C_{ijkl} - t_{ijkl} \f$

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor

  \return symmetric \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//----------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator - (const isotropic_tensor<Tf>& C, const symtensor4<Tf>& t)
//----------------------------------------------------------------------------------------------------

{
  assert (t.dim1() == 3 && t.dim3() == 3);

  symtensor4<Tf> tsr((Tf)-1*t);

  for (int i=1; i<=3; i++)
    { tsr (i,i,i,i) += 2 * C.Mu;
      for (int j=1; j<=3; j++)
	tsr (i,i,j,j) += C.Lambda;
    }

  tsr (1,2,1,2) += C.Mu;
  tsr (1,3,1,3) += C.Mu;
  tsr (2,3,2,3) += C.Mu;

  return tsr;
}


/*!
  \brief Subtraction between a symmetric \f$ \displaystyle 4^{th} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ N \f$ be a symmetric \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = t-C \f$ \n
  \f$ N_{ijkl} = t_{ijkl} - C_{ijkl} \f$.

  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return symmetric \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//----------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator - (const symtensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//----------------------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && t.dim3() == 3);

  symtensor4<Tf> tsr(t);

  for (int i=1; i<=3; i++)
    { tsr (i,i,i,i) -= 2 * C.Mu;
      for (int j=1; j<=3; j++)
	tsr (i,i,j,j) -= C.Lambda;
    }

  tsr (1,2,1,2) -= C.Mu;
  tsr (1,3,1,3) -= C.Mu;
  tsr (2,3,2,3) -= C.Mu;

  return tsr;
}


/*!
  \brief Doubly contracted product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a symmetric \f$ \displaystyle 4^{th} \f$ order tensor.

  We compute \f$ C :: t = C_{ijkl} * t_{ijkl} \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor

  \return scalar \f$ C :: t \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> Tf operator | (const isotropic_tensor<Tf>& C, const symtensor4<Tf>& t)
//----------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && t.dim3() == 3);

  Tf sum=0;

  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      { sum += C.Lambda * t(i,i,j,j);
	sum += 2 * C.Mu * t(i,j,i,j);
      }

  return sum;
}


/*!
  \brief Doubly contracted product between a symmetric \f$ \displaystyle 4^{th} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  We compute \f$ t :: C = C_{ijkl} * t_{ijkl} \f$.

  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return scalar \f$ t :: C \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> Tf operator | (const symtensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//----------------------------------------------------------------------------------------
{
  return (C|t);
}


/*!
  \brief Contracted product between a \f$ \displaystyle 4^{th} \f$ order isotropic tensor and a symmetric \f$ \displaystyle 4^{th} \f$ order tensor.

  Let \f$ N \f$ be a symmetric \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = C : t \f$ \n
  \f$ N_{ijkl} = C_{ijpq} * t_{pqkl} \f$.

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor

  \return symmetric \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//-----------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator || (const isotropic_tensor<Tf>& C, const symtensor4<Tf>& t)
//-----------------------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && t.dim3() == 3);

  symtensor4<Tf> tsr (2 * C.Mu * t);

  for (int i=1; i<=3; i++)
    for (int p=1; p<=3; p++)
      for (int q=p; q<=3; q++)
	for (int k=1; k<=3; k++)
	  tsr (i,i,p,q) += C.Lambda * t (k,k,p,q);

  return tsr;
}


/*!
  \brief Contracted product between a symmetric \f$ \displaystyle 4^{th} \f$ order tensor and a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  Let \f$ N \f$ be a symmetric \f$ \displaystyle 4^{th} \f$ order tensor. We compute \f$ N = t : C \f$ \n
  \f$ N_{ijkl} = t_{ijpq} * C_{pqkl} \f$.

  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return symmetric \f$ \displaystyle 4^{th} \f$ order tensor \f$ N \f$
*/

//-----------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator || (const symtensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//-----------------------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && t.dim3() == 3);

  symtensor4<Tf> tsr (2 * C.Mu * t);

  for (int i=1; i<=3; i++)
    for (int j=i; j<=3; j++)
      for (int p=1; p<=3; p++)
	for (int k=1; k<=3; k++)
	  tsr (i,j,p,p) += C.Lambda * t (i,j,k,k);

  return tsr;
}


/*!
  \brief Compares two \f$ \displaystyle 4^{th} \f$ order isotropic tensors

  We compare the constants \f$ (\lambda_1,\mu_1) \f$ of C1 to \f$ (\lambda_2, \mu_2) \f$ of C2.

  \param C1 \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param C2 \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return TRUE if \f$ C1 = C2 \f$ and FALSE if \f$ C1 \neq C2 \f$
*/

//---------------------------------------------------------------------------------------------------
template <class Tf> bool operator == (const isotropic_tensor<Tf>& C1, const isotropic_tensor<Tf>& C2)
//---------------------------------------------------------------------------------------------------
{
  return(abs(C1.Lambda-C2.Lambda)<epsilon && abs(C1.Mu-C2.Mu)<epsilon);
}


/*!
  \brief Compares two \f$ \displaystyle 4^{th} \f$ order isotropic tensors

  We compare the constants \f$ (\lambda_1,\mu_1) \f$ of C1 to \f$ (\lambda_2, \mu_2) \f$ of C2.

  \param C1 \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param C2 \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return TRUE if \f$ C1 \neq C2 \f$ and FALSE if \f$ C1 = C2 \f$
*/

//---------------------------------------------------------------------------------------------------
template <class Tf> bool operator != (const isotropic_tensor<Tf>& C1, const isotropic_tensor<Tf>& C2)
//---------------------------------------------------------------------------------------------------
{
  return ((C1.Lambda!=C2.Lambda) || (C1.Mu!=C2.Mu));
}


/*!
  \brief Compares a \f$ \displaystyle 4^{th} \f$ order isotropic tensor with a \f$ \displaystyle 4^{th} \f$ order tensor

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t \f$ \displaystyle 4^{th} \f$ order tensor

  \return TRUE if \f$ C = t \f$ and FALSE if \f$ C \neq t \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> bool operator == (const isotropic_tensor<Tf>& C, const tensor4<Tf>& t)
//----------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && (t.dim2() == 3) && (t.dim3() == 3) && (t.dim4() == 3));

  int result = true;
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  result *= (abs(C(i,j,k,l)-t(i,j,k,l))<epsilon);

  return result;
}


/*!
  \brief Compares a \f$ \displaystyle 4^{th} \f$ order tensor with a \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return TRUE if \f$ t = C \f$ and FALSE if \f$ t \neq C \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> bool operator == (const tensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//----------------------------------------------------------------------------------------
{
  return (C == t);
}


/*!
  \brief Compares a \f$ \displaystyle 4^{th} \f$ order isotropic tensor with a \f$ \displaystyle 4^{th} \f$ order tensor

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t \f$ \displaystyle 4^{th} \f$ order tensor

  \return TRUE if \f$ C \neq t \f$ and FALSE if \f$ C = t \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> bool operator != (const isotropic_tensor<Tf>& C, const tensor4<Tf>& t)
//----------------------------------------------------------------------------------------
{
  return !(C == t);
}


/*!
  \brief Compares a \f$ \displaystyle 4^{th} \f$ order tensor with a \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return TRUE if \f$ t \neq C \f$ and FALSE if \f$ t = C \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> bool operator != (const tensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//----------------------------------------------------------------------------------------
{
  return !(C == t);
}


/*!
  \brief Compares a \f$ \displaystyle 4^{th} \f$ order isotropic tensor with a symmetric \f$ \displaystyle 4^{th} \f$ order tensor

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor

  \return TRUE if \f$ C = t \f$ and FALSE if \f$ C \neq t \f$
*/

//-------------------------------------------------------------------------------------------
template <class Tf> bool operator == (const isotropic_tensor<Tf>& C, const symtensor4<Tf>& t)
//-------------------------------------------------------------------------------------------
{
  assert (t.dim1() == 3 && t.dim3() == 3);

  int result = true;

  for (int i=1; i<=3; i++)
    for (int j=i; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=k; l<=3; l++)
	  result *= relative_error (C(i,j,k,l), t(i,j,k,l));

  return result;
}


/*!
  \brief Compares a symmetric \f$ \displaystyle 4^{th} \f$ order tensor with a \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return TRUE if \f$ t = C \f$ and FALSE if \f$ t \neq C \f$
*/

//-------------------------------------------------------------------------------------------
template <class Tf> bool operator == (const symtensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//-------------------------------------------------------------------------------------------
{
  return (C == t);
}


/*!
  \brief Compares a \f$ \displaystyle 4^{th} \f$ order isotropic tensor with a symmetric \f$ \displaystyle 4^{th} \f$ order tensor

  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor
  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor

  \return TRUE if \f$ C \neq t \f$ and FALSE if \f$ C = t \f$
*/

//-------------------------------------------------------------------------------------------
template <class Tf> bool operator != (const isotropic_tensor<Tf>& C, const symtensor4<Tf>& t)
//-------------------------------------------------------------------------------------------
{
  return !(C == t);
}


/*!
  \brief Compares a symmetric \f$ \displaystyle 4^{th} \f$ order tensor with a \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \param t symmetric \f$ \displaystyle 4^{th} \f$ order tensor
  \param C \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \return TRUE if \f$ t \neq C \f$ and FALSE if \f$ t = C \f$
*/

//-------------------------------------------------------------------------------------------
template <class Tf> bool operator != (const symtensor4<Tf>& t, const isotropic_tensor<Tf>& C)
//-------------------------------------------------------------------------------------------
{
  return !(C == t);
}


/*!
  \brief Computes the inverse of a \f$ \displaystyle 4^{th} \f$ order isotropic tensor.

  \f$ \displaystyle C^{-1}_{ijkl} = \frac{1}{4} \mu ( \delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk} ) - \frac{\lambda}{2 \mu (3 \lambda + 2 \mu)} \delta_{ij} \delta_{kl} \f$
  \return inverse tensor
*/

//----------------------------------------------------------------------
template <class T> isotropic_tensor<T> isotropic_tensor<T>::inv () const
//----------------------------------------------------------------------
{
  assert (Mu);
  assert (abs(1.5*Lambda + Mu) > epsilon);

  isotropic_tensor<T> Tinv (-Lambda / (2*Mu * (3*Lambda + 2*Mu)), 0.25/Mu);

  return Tinv;
}


/*!
  \brief Computes the change of basis for a \f$ 4^{th} \f$ order isotropic tensor and an orthogonal change of basis matrix.

  \f$ \displaystyle C^\prime_{pqmn} = \left[ \lambda P_{pi} P_{qi} P_{mj} P_{nj}  + \mu \left( P_{pi} P_{qj} P_{mi} P_{nj} + P_{pi} P_{qj} P_{mj} P_{ni} \right) \right] \f$. \n
  When \f$ P \f$ is an orthogonal matrix (i.e. \f$ P^{-1} = P^t \f$) we get \f$ C^\prime_{pqmn} = C_{pqmn} \f$ .
  \param P orthogonal change of basis matrix
  \param tsr isotropic tensor in the old basis
  \return \f$ C^\prime \f$ isotropic tensor in the new basis
*/

//----------------------------------------------------------------------------------------------------------------------
template <class Tf> isotropic_tensor<Tf> change_orthonormal_basis (const matrix<Tf>& P, const isotropic_tensor<Tf>& tsr)
//----------------------------------------------------------------------------------------------------------------------
{
  assert ((P.Rows() == 3) && (P.Columns() == 3));
  
  return tsr;
}


/*!
  \brief Computes the change of basis for a \f$ 4^{th} \f$ order isotropic tensor.

  \f$ \displaystyle C^\prime_{pqmn} = \left[ \lambda P_{pi} P_{qi} P_{mj} P_{nj}  + \mu \left( P_{pi} P_{qj} P_{mi} P_{nj} + P_{pi} P_{qj} P_{mj} P_{ni} \right) \right] \f$. \n
  \param P change of basis matrix
  \param tsr isotropic tensor in the old basis
  \return \f$ C^\prime \f$ isotropic tensor in the new basis
*/

//-------------------------------------------------------------------------------------------------
template <class Tf> tensor4<Tf> change_basis (const matrix<Tf>& P, const isotropic_tensor<Tf>& tsr)
//-------------------------------------------------------------------------------------------------
{
  assert ((P.Rows() ==3) && (P.Columns() == 3));

  tensor2<Tf> PPt =& (P * t(P)).approximation();
  tensor4<Tf> C =& ( tsr.Lambda * (PPt ^ PPt) );
  for (int p=1; p<=3; p++)
    for (int q=1; q<=3; q++)
      for (int m=1; m<=3; m++)
	for (int n=1; n<=3; n++)
	  C(p,q,m,n) += tsr.Mu * ( PPt(p,m) * PPt(q,n) + PPt(p,n) * PPt(q,m) );

  return C;
}


//------------------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, const isotropic_tensor<Tf>& tsr)
//------------------------------------------------------------------------------------
{
  assert (tsr.Lambda*tsr.Mu);

  s << tsr.tensor_name << endl;
  s << "Material constants : lambda = " << tsr.Lambda << ", mu = " << tsr.Mu << endl;
  s << (tensor4<Tf>) tsr;
  s << endl;

  return s;
}

//------------------------------------------------------------------------------
template <class Tf> istream& operator >> (istream& s, isotropic_tensor<Tf>& tsr)
//------------------------------------------------------------------------------
{
  string text = "";
  tsr.tensor_name = "";
  s >> tsr.Lambda >> tsr.Mu;
  for (int i=1; i<3; i++, text=" ")
    { s >> text;
      if (text != " ")
	{ if (i == 2) tsr.tensor_name += " ";
	  tsr.tensor_name += text;
	  i=1;
	}
    }

  return s;
}


/*!
  \brief Saves on disk a \f$ 4^\text{th} \f$ order isotropic tensor.

  \param path file path and name
*/

//------------------------------------------------------------------
template <class T> void isotropic_tensor<T>::save (const char* path)
//------------------------------------------------------------------
{
  ofstream file(path, ios::out);
  assert (!file.fail());
  file << Lambda << " " << Mu << " " << tensor_name << endl;
  file.close();
}


/*!
  \brief Reads on disk a \f$ 4^\text{th} \f$ order isotropic tensor.

  \param path file path and name
*/

//------------------------------------------------------------------
template <class T> void isotropic_tensor<T>::read (const char* path)
//------------------------------------------------------------------
{
  ifstream file(path, ios::in);
  assert (!file.fail());
  file >> *this;
  file.close();
}


//---------------------------------------------------------------------------
template <class T> isotropic_tensor<T>& isotropic_tensor<T>::approximation ()
//---------------------------------------------------------------------------
{
  assert (Lambda && Mu);
  T elt;

  if (abs(Lambda) < epsilon) Lambda=0;
  if (abs(Mu) < epsilon) Mu=0;

  return *this;
}


}


#endif
